// Generate the ROM data for the translation/display stage of a
// golf club frequency meter.
// Converts count (=time) to frequency to 7-segment display.
//
// Copyright Dave Tutelman  February 1995 -- All rights reserved.

char* usage[] = {
"Usage:  ROMTABLE  <freq-divider stages>  <output stages>  [ <outfile> ]",
"",
"Used to generate the ROM data in a golf-club frequency meter.",
"Generates a data file to go from the binary counter output to the",
"seven-segment display drivers.  Assumes a 1 MHz clock driving the",
"first stage of the counter.  The parameters are:",
"  Freq-divider stages: How many counter stages are there BEFORE any",
"                       stages that drive the display?",
"  Output stages:       How many bits of counter output go to the ROM?",
"  Outfile:             Name of file to dump binary data for the ROM.",
"                       If this optional argument is missing, then the only",
"                       output goes to the screen, and shows the frequencies",
"                       for each count.",
0 };

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>

#define TESTPATTERN

long microsecs;		// How many microseconds in a first-stage count?
long entries;		// How many entries in the table?

// The translation table for the 7-segment display.
// The bit order is:
//	Low order bit is bottom segment.
//	Go around the outside clockwise.
//	High order bit is cross-segment inside.
unsigned char segments[] = {
			0x3F,	// 0
			0x30,	// 1
			0x5B,	// 2
			0x79,	// 3
			0x74,	// 4
			0x6D,	// 5
			0x6F,	// 6
			0x38,	// 7
			0x7F,	// 8
			0x7C,	// 9
			0x76,	// H (need for "HI")
			0x07	// L (need for "LO")
};

const int HI = 10;
const int LO = 11;
const int digits = 4;		// Make it 4 instead of 3 so it's 2^i.
const int hifreq = 999;		// How high with 3 digits?


void help (char*[]);
long freq (int);
void showPat (int, int);
void writeEntry (int,FILE*);


void main (int argc, char** argv)
{
	// Two arguments are first stage and total stages.
	// A good "middling" choice is 9,9.  Gives frequencies
	// starting at 229, and scaled properly to at least 350.
	if (argc<3 || argc>4)
		help (usage);

	int first = atoi (argv[1]);
	int total = atoi (argv[2]);
	if (first<1 || total<1)
		help (usage);

	// Count is in microseconds for a 1 MHz master oscillator.
	// How big is a count in the first stage.
	microsecs = 2 << (first - 1);
	entries   = 2 << (total - 1);

	// Make the table, and show it.
	// We'll show rows of 8 entries.
	for (int i=0; i<entries; i++) {
		if (! (i & 7))
			// Show row designator
			printf("\n%4d: ", i);
		printf(" %8ld", freq(i));
	}
	putchar ('\n');

#ifdef TESTPATTERN
	// Show the characters 0-9 and H and L, in seven-segment form.
	for (i=0; i<3; i++) {
		for (int j=0; j<12; j++) {
			showPat (segments[j], i);
			putchar(' '); putchar(' ');
		}
		putchar('\n');
	}
#endif

	// If optional outfile argument, write bytes to a file.
	if (argc > 3) {
		FILE* f = fopen (argv[3], "wb");
		if (!f) {
			fprintf(stderr,"Can't open file %s for writing\n",
				argv[3]);
			help(usage);
		}

		for (i=0; i<entries; i++)
			writeEntry (i, f);

		fclose (f);

#ifdef TESTPATTERN
		// Let's also show what the display looks like for each entry.
		if ((f = fopen (argv[3], "rb")) == 0) {
			fprintf(stderr,"Can't open file %s for reading\n",
				argv[3]);
			exit (1);
		}

		// Read entries from file and display them, one per lne.
		for (i=0; i<entries; i++) {
			// Read an entry from the file.
			int digit [digits];
			for (int j=0; j<digits; j++)
				digit[j] = getc (f);

			// Show the entry, and the frequency it represents.
			printf("\n%8ld   ", freq(i));
			for (int line=0; line<3; line++) {
				for (j=digits-1; j>=0; j--) {
					showPat (digit[j], line);
					printf("  ");
				}
				printf("\n           ");
			}
		}

		fclose (f);
#endif
	}
}

void
help (char* x[])
{
	for (int i=0; x[i]; i++)
		fprintf(stderr,"%s\n", x[i]);
	exit (1);
}


long
freq (int count)
	// For a given count, what is the frequency?
{
	if (count<1)
		return  -1;

	// The frequency in CPM is 60 million / microseconds counted.
	long time = microsecs * count;
	return  60000000L / time;
}


void
showPat (int b, int line)
	// Show a line (0-2) of the pattern in byte b.
{
	if (line<0 || line>2)
		return;

	switch (line) {
	  case 0:
		putchar (' ');
		putchar ( (b &  8) ? '_' : ' ' );
		putchar (' ');
		break;

	  case 1:
		putchar ( (b &  4) ? '|' : ' ' );
		putchar ( (b & 64) ? '_' : ' ' );
		putchar ( (b & 16) ? '|' : ' ' );
		break;

	  case 2:
		putchar ( (b &  2) ? '|' : ' ' );
		putchar ( (b &  1) ? '_' : ' ' );
		putchar ( (b & 32) ? '|' : ' ' );
		break;

	}
}


void
writeEntry (int n, FILE* f)
	// Write the n-th entry in file f
{
	// Compute the value of the entry.
	long x = freq (n);

	// Display the entry, or HI or LO if it's out of range.
	// The entry will be "digits" bytes, least significant first.
	if (x > hifreq || x < 1) {
		// Too high. Write "HI"
		putc (segments[1], f);
		putc (segments[HI], f);
		for (int i=2; i<digits; i++)
			putc (0, f);

	} else if (n >= entries-1) {
		// Too low. Write "LO"
		putc (segments[0], f);
		putc (segments[LO], f);
		for (int i=2; i<digits; i++)
			putc (0, f);

	} else {
		// Write the frequency entry
		// Break the number into digits.
		int digit;
		for (int i=0; i<digits; i++) {
			digit = (int) x % 10;
			x = x / 10;
			putc (segments [digit], f);
		}
	}
}



